<?php
session_start();
require_once '../config/db.php';
require_once '../auth/auth.php';

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    header('Location: ../auth/login.php');
    exit();
}

// Get sale ID from URL
$sale_id = $_GET['sale_id'] ?? 0;

// Get store information
$stmt = $pdo->prepare("SELECT setting_key, setting_value, setting_binary FROM store_settings WHERE setting_group = 'store_info'");
$stmt->execute();
$settings = [];
$logoData = null;
$logoName = '';
while ($row = $stmt->fetch()) {
    $settings[$row['setting_key']] = $row['setting_value'];
    if ($row['setting_key'] === 'store_logo') {
        $logoData = $row['setting_binary'];
        $logoName = $row['setting_value'];
    }
}

// Debug output to verify store settings
error_log("Store Settings: " . print_r($settings, true));

// Get sale details with customer information
$stmt = $pdo->prepare("
    SELECT s.*, c.name as customer_name, c.phone as customer_phone, c.email as customer_email,
           u.username as cashier_name,
           ct.id as credit_transaction_id, ct.amount as credit_amount,
           COALESCE(SUM(cp.amount), 0) as amount_paid,
           s.total_amount as total,
           s.total_amount as subtotal,
           s.paid_amount as paid_amount
    FROM sales s
    JOIN customers c ON s.customer_id = c.id
    JOIN users u ON s.user_id = u.id
    LEFT JOIN credit_transactions ct ON s.id = ct.sale_id
    LEFT JOIN credit_payments cp ON ct.id = cp.credit_transaction_id
    WHERE s.id = ?
    GROUP BY s.id
");
$stmt->execute([$sale_id]);
$sale = $stmt->fetch();

if (!$sale) {
    die("Sale not found");
}

    // Get sale items
    $stmt = $pdo->prepare("
        SELECT si.*, p.name as product_name, 
               GROUP_CONCAT(psn.serial_number) as serial_numbers
        FROM sale_items si 
        JOIN products p ON si.product_id = p.id 
        LEFT JOIN product_serial_numbers psn ON p.id = psn.product_id 
            AND (
                (psn.status = 'sold' AND psn.source_id = si.sale_id AND psn.source_type = 'manual_adjustment')
                OR 
                (psn.status = 'sold' AND psn.source_id = si.sale_id AND psn.source_type = 'credit_sale')
            )
        WHERE si.sale_id = ?
        GROUP BY si.id
    ");
$stmt->execute([$sale_id]);
$sale_items = $stmt->fetchAll();

// Get credit payments if any
$stmt = $pdo->prepare("
    SELECT cp.*, u.username as received_by
    FROM credit_payments cp
    JOIN users u ON cp.created_by = u.id
    WHERE cp.sale_id = ?
    ORDER BY cp.created_at DESC
");
$stmt->execute([$sale_id]);
$payments = $stmt->fetchAll();

// Get sale payments for payment breakdown
$stmt = $pdo->prepare("SELECT method, amount FROM sale_payments WHERE sale_id = ?");
$stmt->execute([$sale_id]);
$sale_payments = $stmt->fetchAll();

// Calculate remaining balance
$remaining_balance = 0;
if ($sale['credit_transaction_id']) {
    $remaining_balance = $sale['credit_amount'] - $sale['amount_paid'];
}

$receipt_no = $sale['invoice_number'] ?? '';
$date = date('Y-m-d H:i:s', strtotime($sale['created_at']));
$customer_name = $sale['customer_name'] ?? '';
$total = $sale['total'] ?? '';
$paid_amount = $sale['paid_amount'] ?? '';
$payment_method = $sale['payment_method'] ?? '';

// Get currency from settings
$currency = $settings['currency'] ?? 'Ksh';
$website = $settings['website'] ?? '';

// Store info for header
$store_header = [
    'store_name' => $settings['store_name'] ?? '',
    'store_address' => $settings['store_address'] ?? '',
    'store_phone' => $settings['store_phone'] ?? '',
    'store_email' => $settings['store_email'] ?? '',
    'website' => $settings['website'] ?? ''
];
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Print Receipt <?php echo $sale['invoice_number']; ?></title>
    <style>
        @media print {
            @page {
                size: 80mm auto;
                margin: 1;
            }
            body, .receipt {
                font-family: Consolas, monospace !important;
                margin: 0;
                padding: 0;
            }
            .receipt, .receipt * {
                font-weight: bold !important;
            }
            .receipt {
                width: 80mm;
                margin: 0 auto;
            }
            .header {
                text-align: center;
                margin-bottom: 5px;
                padding-bottom: 3px;
                border-bottom: 1px dashed #000;
            }
            .store-logo {
                max-width: 60mm;
                max-height: 30mm;
                margin: 5px auto;
                display: block;
            }
            .store-name {
                font-size: 20px;
                font-weight: 800;
                margin: 10px 0;
                color: #000;
            }
            .store-details {
                font-size: 13px;
                margin: 5px 0;
                padding-bottom: 10px;
                border-bottom: 1px solid #000;
                color: #000;
            }
            .store-details p {
                margin: 3px 0;
                line-height: 1.4;
            }
            .receipt-details {
                font-size: 12px;
                margin-bottom: 10px;
            }
            .items {
                width: 100%;
                margin-bottom: 10px;
            }
            .items th {
                text-align: left;
                border-bottom: 1px solid #000;
                font-size: 12px;
            }
            .items td {
                padding: 2px 0;
            }
            .total {
                margin-top: 10px;
                border-top: 1px dashed #000;
                padding-top: 5px;
                font-size: 12px;
            }
            .footer {
                text-align: center;
                margin-top: 20px;
                font-size: 10px;
            }
            .payment-info {
                margin-top: 10px;
                border-top: 1px dashed #000;
                padding-top: 5px;
                font-size: 12px;
            }
            .payment-row {
                display: flex;
                justify-content: space-between;
                margin: 2px 0;
                font-size: 12px;
            }
            .no-print {
                display: none;
            }
            .dotted-line { 
                border-top: 1px dotted #000; 
                margin: 5px 0; 
            }
            .section-separator { 
                border-top: 1px solid #000; 
                margin: 5px 0; 
            }
            .thankyou { 
                text-align: center; 
                font-weight: bold; 
                font-size: 11px; 
                margin: 4px 0;
                color: black;
            }
            .qr-section { 
                text-align: center; 
                margin: 4px 0; 
            }
            .credit-footer { 
                text-align: center; 
                font-weight: bold; 
                font-size: 11px; 
                margin-top: 5px;
                color: black;
            }
            .info-row {
                font-size: 12px;
            }
        }
        .print-button {
            position: fixed;
            top: 20px;
            right: 20px;
            padding: 10px 20px;
            background: #007bff;
            color: white;
            border: none;
            border-radius: 5px;
            cursor: pointer;
        }
    </style>
</head>
<body>
    <button onclick="window.print()" class="print-button no-print">Print Receipt</button>
    
    <div class="receipt">
        <div class="header">
            <?php if ($logoData): ?>
                <?php
                $mimeType = 'image/jpeg'; // Default to JPEG
                if (strpos($logoName, '.png') !== false) {
                    $mimeType = 'image/png';
                } elseif (strpos($logoName, '.gif') !== false) {
                    $mimeType = 'image/gif';
                }
                ?>
                <img src="data:<?php echo $mimeType; ?>;base64,<?php echo base64_encode($logoData); ?>" alt="Store Logo" class="store-logo">
            <?php endif; ?>
            <div class="store-name"><?php echo htmlspecialchars($store_header['store_name']); ?></div>
            <div class="store-details">
                <p><?php echo htmlspecialchars($store_header['store_address']); ?></p>
                <p>Phone: <?php echo htmlspecialchars($store_header['store_phone']); ?></p>
                <p>Email: <?php echo htmlspecialchars($store_header['store_email']); ?></p>
                <?php if (!empty($store_header['website'])): ?>
                    <p>Web: <?php echo htmlspecialchars($store_header['website']); ?></p>
                <?php endif; ?>
            </div>
        </div>

        <div class="receipt-details">
            Receipt #: <?php echo htmlspecialchars($sale['invoice_number']); ?><br>
            Date: <?php echo date('Y-m-d H:i:s', strtotime($sale['created_at'])); ?><br>
            Cashier: <?php echo htmlspecialchars($sale['cashier_name']); ?><br>
            <?php if ($sale['customer_name']): ?>
                Customer: <?php echo htmlspecialchars($sale['customer_name']); ?><br>
                <?php if ($sale['customer_phone']): ?>
                    Phone: <?php echo htmlspecialchars($sale['customer_phone']); ?><br>
                <?php endif; ?>
            <?php endif; ?>
        </div>

        <table class="items">
            <thead>
                <tr>
                    <th>Item</th>
                    <th>Qty</th>
                    <th>Price</th>
                    <th>Total</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($sale_items as $item): ?>
                <tr>
                    <td style="padding:2px 0;">
                        <?php echo htmlspecialchars($item['product_name']); ?>
                        <?php if (!empty($item['serial_numbers'])): ?>
                            <br><small>SN: <?php echo htmlspecialchars($item['serial_numbers']); ?></small>
                        <?php endif; ?>
                    </td>
                    <td style="text-align:center; padding:2px 0;"><?php echo $item['quantity']; ?></td>
                    <td style="text-align:right; padding:2px 0;"><?php echo $currency . ' ' . number_format($item['price'], 2); ?></td>
                    <td style="text-align:right; padding:2px 0;"><?php echo $currency . ' ' . number_format($item['total'], 2); ?></td>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>

        <div class="total">
            <div class="payment-row">
                <span>Subtotal:</span>
                <span><?php echo number_format($sale['subtotal'], 2); ?></span>
            </div>
            <?php if (isset($sale['discount']) && $sale['discount'] > 0): ?>
            <div class="payment-row">
                <span>Discount:</span>
                <span>-<?php echo number_format($sale['discount'], 2); ?></span>
            </div>
            <?php endif; ?>
            <?php if (isset($sale['shipping']) && $sale['shipping'] > 0): ?>
            <div class="payment-row">
                <span>Shipping:</span>
                <span><?php echo number_format($sale['shipping'], 2); ?></span>
            </div>
            <?php endif; ?>
            <div class="payment-row">
                <span><strong>Total:</strong></span>
                <span><strong><?php echo number_format($sale['total'], 2); ?></strong></span>
            </div>
            <?php if ($sale['payment_method'] === 'cash'): ?>
            <div class="payment-row">
                <span>Cash Tendered:</span>
                <span><?php echo number_format($sale['paid_amount'], 2); ?></span>
            </div>
            <div class="payment-row">
                <span>Change:</span>
                <span><?php echo number_format($sale['paid_amount'] - $sale['total'], 2); ?></span>
            </div>
            <?php endif; ?>
            <?php if ($sale['payment_method'] === 'credit'): ?>
            <div class="payment-row">
                <span>Amount Paid:</span>
                <span><?php echo number_format($sale['paid_amount'], 2); ?></span>
            </div>
            <?php endif; ?>
        </div>

        <?php if ($sale['credit_transaction_id']): ?>
        <div class="payment-info">
            <div class="payment-row">
                <span>Credit Amount:</span>
                <span><?php echo number_format($sale['credit_amount'], 2); ?></span>
            </div>
            <div class="payment-row">
                <span>Amount Paid:</span>
                <span><?php echo number_format($sale['amount_paid'], 2); ?></span>
            </div>
            <div class="payment-row">
                <span><strong>Remaining Balance:</strong></span>
                <span><strong><?php echo number_format($remaining_balance, 2); ?></strong></span>
            </div>

            <?php if (!empty($payments)): ?>
            <div style="margin-top: 10px;">
                <strong>Payment History:</strong><br>
                <?php foreach ($payments as $payment): ?>
                <div style="font-size: 10px; margin-top: 5px;">
                    <?php echo date('Y-m-d H:i', strtotime($payment['created_at'])); ?> - 
                    <?php echo number_format($payment['amount'], 2); ?> 
                    (<?php echo ucfirst($payment['payment_method']); ?>)
                    by <?php echo htmlspecialchars($payment['received_by']); ?>
                </div>
                <?php endforeach; ?>
            </div>
            <?php endif; ?>
        </div>
        <?php endif; ?>

        <?php if ($sale['paid_amount'] < ($total + $sale['shipping'])): ?>
        <div class="info-row">
            <span class="info-label">Remaining:</span>
            <span><?php echo $currency . ' ' . number_format(($total + $sale['shipping']) - $sale['paid_amount'], 2); ?></span>
        </div>
        <?php endif; ?>
        <div class="info-row">
            <span class="info-label">Payment Method:</span>
            <span>
            <?php if (count($payments) > 1): ?>
                Multiple (<?php echo implode(', ', array_map(function($p){return htmlspecialchars($p['payment_method']);}, $payments)); ?>)
            <?php else: ?>
                <?php echo htmlspecialchars($payment_method); ?>
            <?php endif; ?>
            </span>
        </div>
        <div class="thank-you">
          Thank you for shopping with <?php echo htmlspecialchars($store_header['store_name']); ?>
        </div>
        <?php
        // Show unpaid credit items for this customer (if any)
        if (!empty($sale['customer_id'])) {
            $stmt = $pdo->prepare("
                SELECT 
                    ct.id,
                    s.id as sale_id,
                    s.invoice_number,
                    s.created_at as sale_date,
                    ct.amount,
                    COALESCE(SUM(CASE WHEN cp.is_waived = 0 OR cp.is_waived IS NULL THEN cp.amount ELSE 0 END), 0) as paid,
                    COALESCE(SUM(CASE WHEN cp.is_waived = 1 THEN cp.amount ELSE 0 END), 0) as waived
                FROM credit_transactions ct 
                JOIN sales s ON ct.sale_id = s.id 
                LEFT JOIN credit_payments cp ON ct.id = cp.credit_transaction_id
                WHERE ct.customer_id = ? 
                GROUP BY ct.id 
                HAVING ct.amount - paid - waived > 0 
                ORDER BY s.created_at ASC
            ");
            $stmt->execute([$sale['customer_id']]);
            $unpaid_credits = $stmt->fetchAll();
            if (!empty($unpaid_credits)) {
                echo '<div style="border-top: 2px solid #000; margin: 5px 0;"></div>';
                echo '<div style="border-top: 2px solid #000; margin: 2px 0;"></div>';
                echo '<div style="font-weight:bold; text-align:center; margin-bottom:4px;">UNPAID CREDIT ITEMS</div>';
                
                foreach ($unpaid_credits as $credit) {
                    $balance = $credit['amount'] - $credit['paid'] - $credit['waived'];
                    
                    // Get items for this sale
                    $stmt = $pdo->prepare("
                        SELECT si.*, p.name as product_name, 
                               GROUP_CONCAT(psn.serial_number) as serial_numbers
                        FROM sale_items si 
                        JOIN products p ON si.product_id = p.id 
                        LEFT JOIN product_serial_numbers psn ON p.id = psn.product_id 
                            AND psn.status = 'sold' 
                            AND psn.source_id = si.sale_id
                        WHERE si.sale_id = ?
                        GROUP BY si.id
                    ");
                    $stmt->execute([$credit['sale_id']]);
                    $sale_items = $stmt->fetchAll();
                    
                    // Invoice header
                    echo '<div style="margin-bottom:8px; border:1px solid #000; padding:4px;">';
                    echo '<div style="font-weight:bold; font-size:11px; text-align:center; border-bottom:1px solid #000; padding-bottom:2px;">';
                    echo 'Invoice #: ' . htmlspecialchars($credit['invoice_number']) . '<br>';
                    echo 'Date: ' . date('M d, Y', strtotime($credit['sale_date']));
                    if ($credit['waived'] > 0) {
                        echo '<br>Waived: ' . $currency . ' ' . number_format($credit['waived'], 2);
                    }
                    echo '</div>';
                    
                    // Items table
                    echo '<table style="width:100%; font-size:10px; border-collapse:collapse; margin:2px 0;">';
                    echo '<thead><tr>';
                    echo '<th style="text-align:left; padding:2px;">Item</th>';
                    echo '<th style="text-align:right; padding:2px;">Amount</th>';
                    echo '<th style="text-align:right; padding:2px;">Paid</th>';
                    echo '<th style="text-align:right; padding:2px;">Waived</th>';
                    echo '<th style="text-align:right; padding:2px;">Balance</th>';
                    echo '</tr></thead>';
                    echo '<tbody>';
                    echo '<tr>';
                    echo '<td style="padding:2px;">Credit Sale</td>';
                    echo '<td style="text-align:right; padding:2px;">' . $currency . ' ' . number_format($credit['amount'], 2) . '</td>';
                    echo '<td style="text-align:right; padding:2px;">' . $currency . ' ' . number_format($credit['paid'], 2) . '</td>';
                    echo '<td style="text-align:right; padding:2px;">' . $currency . ' ' . number_format($credit['waived'], 2) . '</td>';
                    echo '<td style="text-align:right; padding:2px;">' . $currency . ' ' . number_format($balance, 2) . '</td>';
                    echo '</tr>';
                    echo '</tbody>';
                    echo '</table>';
                    echo '</div>';
                    echo '<div class="dotted-line"></div>';
                }
            }
        }
        ?>
        <?php
        // Gather all receipt info for QR
        $qr_lines = [];
        $qr_lines[] = $store_header['store_name'];
        $qr_lines[] = $store_header['store_address'];
        $qr_lines[] = 'Tel: ' . $store_header['store_phone'];
        $qr_lines[] = 'Email: ' . $store_header['store_email'];
        if (!empty($store_header['website'])) {
            $qr_lines[] = $store_header['website'];
        }
        $qr_lines[] = "Receipt #: {$receipt_no}";
        $qr_lines[] = "Date: {$date}";
        $qr_lines[] = "Client: " . ($customer_name ?? 'Walk-in');
        $qr_lines[] = "Cashier: " . ($sale['cashier_name'] ?? 'Unknown');
        $qr_lines[] = "Total: {$currency} " . number_format($total + $sale['shipping'], 2);
        
        // Add detailed payment breakdown to QR code
        if (count($sale_payments) > 1) {
            $qr_lines[] = 'Payment Breakdown:';
            $actual_paid_amount = 0; // Track only cash/mpesa amounts
            foreach ($sale_payments as $payment) {
                $method = strtolower($payment['method']);
                if ($method === 'cash' || $method === 'mpesa') {
                    $actual_paid_amount += $payment['amount'];
                }
                $qr_lines[] = ucfirst($payment['method']) . ': ' . $currency . ' ' . number_format($payment['amount'], 2);
            }
            $qr_lines[] = 'Total Paid: ' . $currency . ' ' . number_format($actual_paid_amount, 2);
        } else {
            // For single payment, check if it's cash/mpesa or credit
            $payment = $sale_payments[0] ?? null;
            if ($payment && (strtolower($payment['method']) === 'cash' || strtolower($payment['method']) === 'mpesa')) {
                $qr_lines[] = 'Amount Paid: ' . $currency . ' ' . number_format($payment['amount'], 2);
            } else {
                $qr_lines[] = 'Amount Paid: ' . $currency . ' ' . number_format(0, 2); // Credit sale, no immediate payment
            }
        }
        
        $qr_lines[] = "Payment: " . (count($payments) > 1 ? 'Multiple' : $payment_method);
        $qr_data = implode("\n", $qr_lines);
        $qr_url = 'https://api.qrserver.com/v1/create-qr-code/?size=150x150&data=' . urlencode($qr_data);
        ?>
        <div class="qr-section">
          <img src="<?php echo $qr_url; ?>" alt="QR Code" style="width:150px; height:150px;" />
        </div>
        <div class="dotted-line"></div>
        <div class="credit-footer">
          Designed And Developed By Vico Technologies
        </div>
    </div>

    <script>
    window.onload = function() {
        window.print();
    };
    window.onafterprint = function() {
        if (window.opener) {
            window.opener.location.href = '../pages/reports.php';
            window.close();
        } else if (window.parent) {
            window.parent.location.href = '../pages/reports.php';
            window.close();
        } else {
            window.location.href = '../pages/reports.php';
        }
    };
    </script>
</body>
</html> 